﻿using System.Collections.Generic;
using Newtonsoft.Json;
using Oxide.Core;
using Rust;
using UnityEngine;

namespace Oxide.Plugins
{
    [Info("ExplodingBarrels", "db_arcane", "1.0.1")]
    [Description("Oil barrels explode when hit by a bullet, like in every other game made since the beginning of time.")]
    class ExplodingBarrels : RustPlugin
    {
        private bool IsBullet(HitInfo hitInfo)
        {
            // confirm we have a valid hitInfo
            if (hitInfo == null) 
                return false;

            // confirm we have a projectile
            if (!hitInfo.IsProjectile())
                return false;

            // if not a bullet, return false
            if (!hitInfo.ProjectilePrefab.ToString().Contains("bullet"))
                return false;

            // if a snowball, return false
            if (hitInfo.ProjectilePrefab.ToString().Contains("snowball"))
                return false;

            // otherwise, 
            return true;
        }

        object OnEntityTakeDamage(BaseCombatEntity entity, HitInfo hitInfo)
        {
            // if not an oil barrel, return
            if (entity.ShortPrefabName != "oil_barrel")
                return null;

            // if no initiating player, return
            if (hitInfo.InitiatorPlayer == null)
                return null;

            // explode the barrel if it was shot or blown up (by another oil barrel, to allow a chain reaction)
            if ((IsBullet(hitInfo)) || (hitInfo.damageTypes.GetMajorityDamageType() == DamageType.Explosion))
            {
                // set the position of the explosion to the barrel's, plus 1/2 a meter up
                var pos = entity.transform.position;
                pos.y += 0.5f;

                // set the damage type for the explosion
                DamageTypeEntry dmg = new DamageTypeEntry();
                dmg.type = DamageType.Explosion;
                dmg.amount = 100;

                // prep the explosion and spawn
                TimedExplosive grenade = GameManager.server.CreateEntity("assets/prefabs/weapons/f1 grenade/grenade.f1.deployed.prefab") as TimedExplosive;
                grenade.ServerPosition = pos;
                grenade.SetCreatorEntity(hitInfo.InitiatorPlayer);

                grenade.damageTypes.Clear();
                grenade.damageTypes.Add(dmg);

                grenade.Spawn();

                // give it a little bit of a delay so that when there's a chain reaction, it's more noticeable
                grenade.SetFuse(0.1f);

                // kill the barrel and its contents
                entity.Kill();

                return false;
            }

            // otherwise, 
            return null;
        }
    }
}
