using HarmonyLib;
using Newtonsoft.Json;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using UnityEngine;
using UnityEngine.SceneManagement;

namespace Oxide.Plugins
{
    [Info("GhostRider", "0xF", "1.1.0")]
    partial class GhostRider : RustPlugin
    {
        private const string MOTORBIKE_DRIVER_SEAT_PREFAB = "assets/prefabs/vehicle/seats/motorbikedriverseat.prefab";
        private const string BOOST_EFFECT_PREFAB = "assets/prefabs/npc/patrol helicopter/effects/gun_fire_small.prefab";
        private const string FLAME_PREFAB = "assets/prefabs/weapons/flamethrower/flamethrower_fireball.prefab";
        private const uint FIREBALL_PREFABID = 3369311876;
        private static uint HEADBONE = StringPool.Get("head");
        private static GhostRider Instance;
        private Dictionary<Bike, List<BaseEntity>> subEntities = new Dictionary<Bike, List<BaseEntity>>();

        void Init()
        {
            Instance = this;
        }

        void OnServerInitialized()
        {
            Bike.doPlayerDamage = config.CanCrashDamage;

            foreach (BasePlayer player in BasePlayer.allPlayerList)
            {
                DisposeRiderStaff(player);
            }

            foreach (Bike bike in BaseNetworkable.serverEntities.OfType<Bike>())
            {
                if (bike.HasDriver())
                    SetupRiderStaff(bike.GetDriver(), bike);
            }
        }

        void OnEntityKill(Bike bike)
        {
            DisposeRiderStaff(bike);
        }

        void Unload()
        {
            foreach (BasePlayer player in BasePlayer.allPlayerList)
                DisposeRiderStaff(player);
            foreach (Bike bike in BaseNetworkable.serverEntities.OfType<Bike>())
                DisposeRiderStaff(bike);
        }

        private BaseEntity CreateFireball()
        {
            GameObject gameObject = new GameObject("ghostrider-fireball");
            gameObject.layer = (int)Rust.Layer.Reserved1;
            SceneManager.MoveGameObjectToScene(gameObject, Rust.Server.EntityScene);
            BaseEntity fireball = gameObject.AddComponent<BaseEntity>();
            fireball.prefabID = 3369311876;
            fireball.enableSaving = false;
            gameObject.SetActive(true);
            return fireball;
        }

        private static bool IsValidBike(Bike bike)
        {
            if (!bike)
                return false;

            if (!bike.PrefabName.Contains("motor"))
                return false;

            if (config.Only2Wheels && bike.GetWheels().Length != 2)
                return false;
            return true;
        }

        private static bool IsMotorbikeSeat(BikeDriverSeat seat)
        {
            return seat.PrefabName == MOTORBIKE_DRIVER_SEAT_PREFAB;
        }

        private void SetupRiderStaff(BasePlayer player, Bike bike)
        {
            if (!IsValidBike(bike))
                return;

            if (subEntities.TryGetValue(bike, out List<BaseEntity> ents))
            {
                foreach (var ent in ents)
                    if (!ent.IsDestroyed)
                        ent.Kill();
                ents.Clear();
            }
            else
            {
                ents = subEntities[bike] = new List<BaseEntity>();
            }

            foreach (CarWheel wheel in bike.GetWheels())
            {
                BaseEntity fireball = CreateFireball();
                fireball.SetParent(bike);
                fireball.transform.localPosition = wheel.wheelCollider.transform.localPosition;
                fireball.Spawn();
                ents.Add(fireball);
            }

            BaseEntity headFireball = CreateFireball();
            headFireball.SetParent(player, HEADBONE);
            headFireball.Spawn();
            ents.Add(headFireball);

            bike.gameObject.AddComponent<GroundFlame>();
        }

        private void DisposeRiderStaff(Bike bike)
        {
            if (subEntities.TryGetValue(bike, out List<BaseEntity> ents))
            {
                foreach (var ent in ents)
                    if (!ent.IsDestroyed)
                        ent.Kill();
                ents.Clear();
                UnityEngine.Object.Destroy(bike.gameObject.GetComponent<GroundFlame>());
            }
        }

        private class GroundFlame : FacepunchBehaviour
        {
            private Bike bike;
            private float tickRate = 0.15f;
            private float nextFlameTime;
            private float range = 30f;

            void Start()
            {
                bike = GetComponent<Bike>();
                InvokeRepeating(new Action(this.FlameTick), 0.15f, 0.15f);
            }
            public void FlameTick()
            {
                if (bike.GetSpeed() > 5 && UnityEngine.Time.realtimeSinceStartup >= this.nextFlameTime)
                {
                    this.nextFlameTime = UnityEngine.Time.realtimeSinceStartup + 0.4f;
                    FireBall fireball = global::GameManager.server.CreateEntity(FLAME_PREFAB, bike.transform.position - bike.transform.rotation * -Vector3.forward * 2, default(Quaternion), true) as FireBall;
                    if (fireball)
                    {
                        fireball.creatorEntity = bike.GetDriver();
                        fireball.lifeTimeMin = 3;
                        fireball.lifeTimeMax = 3;
                        fireball.Spawn();
                        fireball.CancelInvoke(new Action(fireball.Think));
                    }
                }
            }
        }

        private void DisposeRiderStaff(BasePlayer player)
        {
            foreach (BaseEntity fireball in player.children.Where(e => e.prefabID == FIREBALL_PREFABID && e.parentBone == HEADBONE).ToArray())
            {
                if (!fireball.IsDestroyed)
                    fireball.Kill();
            }
        }


        void Loaded()
        {
            foreach (Type type in this.GetType().GetNestedTypes(BindingFlags.DeclaredOnly | BindingFlags.Static | BindingFlags.Public | BindingFlags.NonPublic))
            {
                object[] attribute = type.GetCustomAttributes(typeof(HarmonyPatch), false);
                if (attribute.Length >= 1)
                {
                    PatchClassProcessor patchClassProcessor = this.HarmonyInstance.CreateClassProcessor(type);
                    patchClassProcessor.Patch();
                }
            }
        }


        [HarmonyPatch(typeof(BaseVehicle), "PlayerMounted")]
        private static class BaseVehicle_PlayerMounted_Patch
        {
            private static bool Prefix(BaseVehicle __instance, BasePlayer __0, BaseMountable __1)
            {
                if (__instance is Bike && __1 is BikeDriverSeat && IsMotorbikeSeat(__1 as BikeDriverSeat))
                {
                    Instance.SetupRiderStaff(__0, __instance as Bike);
                    return false;
                }
                return true;
            }
        }

        [HarmonyPatch(typeof(BaseVehicle), "PrePlayerDismount")]
        private static class BaseVehicle_PrePlayerDismount_Patch
        {
            private static bool Prefix(BaseVehicle __instance, BaseMountable __1)
            {
                if (__instance is Bike && __1 is BikeDriverSeat && IsMotorbikeSeat(__1 as BikeDriverSeat))
                {
                    Instance.DisposeRiderStaff(__instance as Bike);
                    return false;
                }
                return true;
            }
        }

        [HarmonyPatch(typeof(BasePlayer), "DismountObject")]
        private static class BasePlayer_DismountObject_Patch
        {
            private static void Postfix(BasePlayer __instance)
            {
                Instance.DisposeRiderStaff(__instance);
            }
        }

        [HarmonyPatch(typeof(Bike), "PlayerServerInput")]
        private static class Bike_PlayerServerInput_Patch
        {
            private static void Postfix(Bike __instance, InputState inputState, BasePlayer player)
            {
                if (!__instance.IsDriver(player) || !IsValidBike(__instance))
                    return;

                if (inputState.WasJustPressed(BUTTON.SPRINT))
                {
                    Effect.server.Run(BOOST_EFFECT_PREFAB, __instance.transform.position + __instance.transform.rotation * new Vector3(0.2f, 1f, -0.7f));
                }
            }
        }

        [HarmonyPatch(typeof(Bike), "GetMaxDriveForce")]
        private static class Bike_GetMaxDriveForce_Patch
        {
            private static void Postfix(Bike __instance, ref float __result)
            {
                if (__instance.SprintInput && IsValidBike(__instance))
                    __result = __result * config.EngineMul;
            }
        }

        #region Config
        static Configuration config;
        public class Configuration
        {
            [JsonProperty(PropertyName = "Only 2-wheels motorbike?")]
            public bool Only2Wheels { get; set; } = false;
            [JsonProperty(PropertyName = "Multiplier Engine Power")]
            public float EngineMul { get; set; } = 2.5f;
            [JsonProperty(PropertyName = "Can bike crashes cause damage or death to the rider?")]
            public bool CanCrashDamage { get; set; } = false;

            public static Configuration DefaultConfig()
            {
                return new Configuration();
            }
        }

        protected override void LoadConfig()
        {
            base.LoadConfig();
            try
            {
                config = Config.ReadObject<Configuration>();
                if (config == null) LoadDefaultConfig();
                SaveConfig();
            }
            catch (Exception ex)
            {
                Debug.LogException(ex);
                PrintWarning("Creating new configuration file.");
                LoadDefaultConfig();
            }
        }

        protected override void LoadDefaultConfig() => config = Configuration.DefaultConfig();
        protected override void SaveConfig() => Config.WriteObject(config);
        #endregion
    }
}