﻿using Oxide.Core.Plugins;
using System;
using System.Collections.Generic;
using UnityEngine;

namespace Oxide.Plugins
{
    [Info("OneHitKillAnimals", "HAVOC805", "2.0.0")]
    [Description("One hit kill for animals")]
    public class OneHitKillAnimals : RustPlugin
    {
        // Updated 2025-02-06 04:36:14 by HAVOC805
        // Wolf2 is handled separately to avoid server crashes

        private Dictionary<string, bool> _animalOneHitConfig = new Dictionary<string, bool>(StringComparer.OrdinalIgnoreCase);
        private bool _includeGuns = true;
        private const string UsePermission = "onehitkillanimals.use";
        private const string CommandPermission = "onehitkillanimals.commands";

        protected override void LoadDefaultConfig()
        {
            // Default config - everything enabled
            Config["AnimalOneHitConfig"] = new Dictionary<string, string>
            {
                { "Bear", "true" },
                { "Boar", "true" },
                { "Chicken", "true" },
                { "Deer", "true" },
                { "Horse", "true" },
                { "Wolf", "true" },
                { "Stag", "true" },
                { "Shark", "true" }
            };
            Config["IncludeGuns"] = "true";
        }

        // Load settings from config
        private new void LoadConfig()
        {
            _animalOneHitConfig.Clear();
            var configAnimals = Config.Get<Dictionary<string, string>>("AnimalOneHitConfig");
            if (configAnimals != null)
            {
                foreach (var kvp in configAnimals)
                {
                    bool enabled;
                    if (bool.TryParse(kvp.Value, out enabled))
                        _animalOneHitConfig[kvp.Key] = enabled;
                }
            }
            _includeGuns = Config.Get<string>("IncludeGuns") == "true";
        }

        void Init()
        {
            // Set up permissions
            permission.RegisterPermission(UsePermission, this);
            permission.RegisterPermission(CommandPermission, this);
            LoadConfig();
        }

        // Main damage hook
        object OnEntityTakeDamage(BaseCombatEntity entity, HitInfo hitInfo)
        {
            try
            {
                if (entity == null || hitInfo?.Initiator == null) return null;
                if (!(hitInfo.Initiator is BasePlayer player)) return null;
                if (!permission.UserHasPermission(player.UserIDString, UsePermission)) return null;

                // Wolf2 gets special treatment - always one hit
                if (entity.ShortPrefabName == "wolf2" || entity.name.Contains("wolf2"))
                {
                    hitInfo.damageTypes.ScaleAll(100f);
                    return null;
                }

                // Handle other animals if they're enabled
                if (entity is BaseNpc)
                {
                    string type = GetAnimalType(entity);
                    bool enabled;
                    if (_animalOneHitConfig.TryGetValue(type, out enabled) && enabled)
                    {
                        // Skip if guns aren't allowed
                        if (!_includeGuns && hitInfo.Weapon is BaseProjectile)
                            return null;

                        hitInfo.damageTypes.ScaleAll(100f);
                    }
                }
            }
            catch (Exception ex)
            {
                // Log error but don't crash
                Puts($"Error in damage hook: {ex.Message}");
            }
            return null;
        }

        // Quick check what kind of animal we're dealing with
        private string GetAnimalType(BaseCombatEntity entity)
        {
            if (entity == null) return "Unknown";

            string name = (entity.PrefabName ?? "").ToLower();

            // Simple if-else chain, easier to read than a complex switch
            if (name.Contains("/bear")) return "Bear";
            if (name.Contains("/boar")) return "Boar";
            if (name.Contains("/chicken")) return "Chicken";
            if (name.Contains("/stag") || name.Contains("/deer")) return "Deer";
            if (name.Contains("/horse")) return "Horse";
            if (name.Contains("/wolf")) return "Wolf";
            if (name.Contains("/shark")) return "Shark";

            return "Unknown";
        }

        [ChatCommand("toggleonehitkill")]
        private void CmdToggleOneHitKill(BasePlayer player, string command, string[] args)
        {
            if (!permission.UserHasPermission(player.UserIDString, CommandPermission))
            {
                player.ChatMessage("You don't have permission to use this command.");
                return;
            }

            // Show settings if no args
            if (args.Length == 0)
            {
                ShowCurrentSettings(player);
                return;
            }

            string setting = args[0].ToLower();
            if (setting == "includeguns")
                ToggleIncludeGuns(player, args);
            else if (_animalOneHitConfig.ContainsKey(setting))
                ToggleAnimalSetting(player, setting, args);
            else
                ShowCurrentSettings(player);
        }

        private void ShowCurrentSettings(BasePlayer player)
        {
            player.ChatMessage("OneHitKillAnimals Settings:");
            player.ChatMessage("Wolf2: Always on (can't be turned off)");
            player.ChatMessage($"Guns: {_includeGuns}");

            // Show other animals
            foreach (var animal in _animalOneHitConfig)
                player.ChatMessage($"{animal.Key}: {animal.Value}");

            player.ChatMessage("Use: /toggleonehitkill <animal|guns> <true|false>");
        }

        private void ToggleIncludeGuns(BasePlayer player, string[] args)
        {
            if (args.Length < 2)
            {
                player.ChatMessage($"Guns currently: {_includeGuns}");
                return;
            }

            bool newValue;
            if (bool.TryParse(args[1], out newValue))
            {
                _includeGuns = newValue;
                Config["IncludeGuns"] = newValue.ToString();
                SaveConfig();
                player.ChatMessage($"Guns now {(newValue ? "enabled" : "disabled")}");
            }
            else
                player.ChatMessage("Use true or false");
        }

        private void ToggleAnimalSetting(BasePlayer player, string setting, string[] args)
        {
            if (args.Length < 2)
            {
                player.ChatMessage($"{setting} is currently: {_animalOneHitConfig[setting]}");
                return;
            }

            bool newValue;
            if (bool.TryParse(args[1], out newValue))
            {
                // Update both memory and config
                _animalOneHitConfig[setting] = newValue;

                var configDict = new Dictionary<string, string>();
                foreach (var item in _animalOneHitConfig)
                    configDict[item.Key] = item.Value.ToString();

                Config["AnimalOneHitConfig"] = configDict;
                SaveConfig();

                player.ChatMessage($"{setting} now {(newValue ? "enabled" : "disabled")}");
            }
            else
                player.ChatMessage("Use true or false");
        }
    }
}