using Newtonsoft.Json;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using Rust;
using UnityEngine;
using UnityEngine.Networking;
using System.Threading.Tasks;
using System.Text;
using Oxide.Core;
using Newtonsoft.Json.Linq;

//  ____  _       _ _       
// / ___|(_) __ _(_) | ___  
// \___ \| |/ _` | | |/ _ \ 
//  ___) | | (_| | | | (_) |
// |____/|_|\__, |_|_|\___/ 
//          |___/             
// ☽✧✵✧✵✧✵✧✵✧✵✧✵✧☾
// ✦    RUST PLUGINS   ✦
// ✦     Sigilo.dev    ✦
// ☽✧✵✧✵✧✵✧✵✧✵✧✵✧☾

namespace Oxide.Plugins
{
    [Info("RustAI", "Sigilo", "1.3.2")]
    class RustAI : RustPlugin
    {
        private PluginConfig _config { get; set; }
        private Dictionary<string, float> _lastUsageTime = new Dictionary<string, float>();
        private float _lastGlobalUsageTime;
        private Dictionary<string, List<ConversationEntry>> _conversationHistory = new Dictionary<string, List<ConversationEntry>>();
        private const string ConversationDirectory = "data/RustAI/Conversations";
        private Dictionary<string, List<ConversationEntry>> _cachedConversations = new Dictionary<string, List<ConversationEntry>>();
        private Dictionary<string, bool> _mutedPlayers = new Dictionary<string, bool>();

        private const string UsePermission = "rustai.use";
        private const string AdminPermission = "rustai.admin";
        private const string ToggleBotPermission = "rustai.toggle";

        public class ConversationEntry
        {
            public string PlayerName { get; set; }
            public string PlayerSteamId { get; set; }
            public string Question { get; set; }
            public string Response { get; set; }
            public DateTime Timestamp { get; set; }
        }

        public class ServerInfo
        {
            public string MaxTeamSize { get; set; }
            public string WipeSchedule { get; set; }
            public string DiscordInfo { get; set; }
            public string Website { get; set; }
            public Dictionary<string, string> CustomInfo { get; set; }
        }

        public class PluginConfig
        {
            public string OpenAIApiURL { get; set; }
            public string TextGenerationApiUrl { get; set; }
            public List<string> ActivationKeywords { get; set; }
            public float UserCooldownInSeconds { get; set; }
            public float GlobalCooldownInSeconds { get; set; }
            public string SystemPrompt { get; set; }
            public string ModelType { get; set; }
            public string OpenAI_API_Key { get; set; }
            public string ModelName { get; set; }
            public int MaxTokens { get; set; }
            public double Temperature { get; set; }
            public string Character { get; set; }
            public string CharacterColor { get; set; }
            public string DiscordWebhookURL { get; set; }
            public bool SendCooldownMessages { get; set; }
            public ServerInfo ServerInformation { get; set; }
            public string ResponseLanguage { get; set; }
            public string EmptyPromptTemplate { get; set; }
            public string CooldownMessageTemplate { get; set; }
            public string NoPermissionMessage { get; set; }
            public string ChatFormat { get; set; }
            public bool UseUncensoredMode { get; set; }
            public string UncensoredModePrompt { get; set; }
            public string CensoredModePrompt { get; set; }
            public List<string> IllegalTopics { get; set; }
            public int ConversationMemorySize { get; set; }
            public int TimeZoneOffset { get; set; }

            public static PluginConfig DefaultConfig()
            {
                return new PluginConfig
                {
                    OpenAIApiURL = "https://api.openai.com/v1/chat/completions",
                    TextGenerationApiUrl = "http://0.0.0.0:5000/v1/chat/completions",
                    ActivationKeywords = new List<string> { "!bot", "!admin" },
                    SystemPrompt = "You are a helpful assistant for a Rust game server. Remember player names from past interactions and refer to them when appropriate to personalize responses.",
                    OpenAI_API_Key = "your openai api key here",
                    DiscordWebhookURL = "your discord webhook here",
                    UserCooldownInSeconds = 60.0f,
                    GlobalCooldownInSeconds = 10.0f,
                    ModelType = "openai",
                    ModelName = "gpt-4o-mini",
                    MaxTokens = 100,
                    Temperature = 1.0,
                    Character = "Server Assistant",
                    CharacterColor = "#7FFF00",
                    SendCooldownMessages = true,
                    ServerInformation = new ServerInfo
                    {
                        MaxTeamSize = "TRIO",
                        WipeSchedule = "Thursdays at 4 PM EST",
                        DiscordInfo = "discord.gg/yourserver",
                        Website = "yourwebsite.com",
                        CustomInfo = new Dictionary<string, string>
                        {
                            { "PvP", "Enabled at all times" },
                            { "Loot", "1.5x" },
                            { "Resources", "2x" }
                        }
                    },
                    ResponseLanguage = "English",
                    EmptyPromptTemplate = "What can I help you with? You sent an empty message.",
                    CooldownMessageTemplate = "Please wait {0} seconds before asking again.",
                    NoPermissionMessage = "You don't have permission to use this command.",
                    ChatFormat = "<color={0}>{1}</color>: {2}",
                    UseUncensoredMode = true,
                    UncensoredModePrompt = "IMPORTANT: You are running in UNCENSORED mode. Be authentic to the raw Rust gaming environment. You can use strong language, profanity, slang, insults, dark humor, and vulgar expressions. Be direct, crude and match the harsh atmosphere of Rust. Don't hold back except for illegal topics.",
                    CensoredModePrompt = "You are running in censored mode. Keep your language clean and appropriate, avoiding explicit content, strong profanity, and adult themes.",
                    IllegalTopics = new List<string>
                    {
                        "doxxing",
                        "dangerous illegal activities"
                    },
                    ConversationMemorySize = 5,
                    TimeZoneOffset = 0
                };
            }

            public void EnsureDefaultValues()
            {
                var defaultConfig = DefaultConfig();
                
                if (OpenAIApiURL == null)
                    OpenAIApiURL = defaultConfig.OpenAIApiURL;
                    
                if (TextGenerationApiUrl == null)
                    TextGenerationApiUrl = defaultConfig.TextGenerationApiUrl;
                    
                if (ActivationKeywords == null)
                    ActivationKeywords = defaultConfig.ActivationKeywords;
                    
                if (UserCooldownInSeconds <= 0)
                    UserCooldownInSeconds = defaultConfig.UserCooldownInSeconds;
                    
                if (GlobalCooldownInSeconds <= 0)
                    GlobalCooldownInSeconds = defaultConfig.GlobalCooldownInSeconds;
                    
                if (string.IsNullOrEmpty(SystemPrompt))
                    SystemPrompt = defaultConfig.SystemPrompt;
                    
                if (string.IsNullOrEmpty(ModelType))
                    ModelType = defaultConfig.ModelType;
                    
                if (string.IsNullOrEmpty(OpenAI_API_Key))
                    OpenAI_API_Key = defaultConfig.OpenAI_API_Key;
                    
                if (string.IsNullOrEmpty(ModelName))
                    ModelName = defaultConfig.ModelName;
                    
                if (MaxTokens <= 0)
                    MaxTokens = defaultConfig.MaxTokens;
                    
                if (Temperature <= 0)
                    Temperature = defaultConfig.Temperature;
                    
                if (string.IsNullOrEmpty(Character))
                    Character = defaultConfig.Character;
                    
                if (string.IsNullOrEmpty(CharacterColor))
                    CharacterColor = defaultConfig.CharacterColor;
                    
                if (ServerInformation == null)
                    ServerInformation = defaultConfig.ServerInformation;
                else
                {
                    if (string.IsNullOrEmpty(ServerInformation.MaxTeamSize))
                    {
                        ServerInformation.MaxTeamSize = defaultConfig.ServerInformation.MaxTeamSize;
                    }
                    
                    if (ServerInformation.CustomInfo == null)
                        ServerInformation.CustomInfo = defaultConfig.ServerInformation.CustomInfo;
                }
                
                if (string.IsNullOrEmpty(ResponseLanguage))
                    ResponseLanguage = defaultConfig.ResponseLanguage;
                    
                if (string.IsNullOrEmpty(EmptyPromptTemplate))
                    EmptyPromptTemplate = defaultConfig.EmptyPromptTemplate;
                    
                if (string.IsNullOrEmpty(CooldownMessageTemplate))
                    CooldownMessageTemplate = defaultConfig.CooldownMessageTemplate;
                    
                if (string.IsNullOrEmpty(NoPermissionMessage))
                    NoPermissionMessage = defaultConfig.NoPermissionMessage;
                    
                if (string.IsNullOrEmpty(ChatFormat))
                    ChatFormat = defaultConfig.ChatFormat;
                    
                if (string.IsNullOrEmpty(UncensoredModePrompt))
                    UncensoredModePrompt = defaultConfig.UncensoredModePrompt;
                    
                if (string.IsNullOrEmpty(CensoredModePrompt))
                    CensoredModePrompt = defaultConfig.CensoredModePrompt;
                    
                if (IllegalTopics == null)
                    IllegalTopics = defaultConfig.IllegalTopics;
                    
                if (ConversationMemorySize <= 0)
                    ConversationMemorySize = defaultConfig.ConversationMemorySize;
                
                if (TimeZoneOffset < -12 || TimeZoneOffset > 14)
                    TimeZoneOffset = defaultConfig.TimeZoneOffset;
            }
        }

        public class Payload
        {
            public string prompt { get; set; }
            public int max_tokens { get; set; }
            public double temperature { get; set; }
        }

        public class Response
        {
            public string id { get; set; }
            public string created { get; set; }
            public string model { get; set; }
            public Choice[] choices { get; set; }
        }

        public class Choice
        {
            public Message message { get; set; }
            public string text { get; set; }
        }

        public class Message
        {
            public string content { get; set; }
        }

        public class ServerMessage
        {
            public string @event { get; set; }
            public string text { get; set; }
        }

        private string GetConversationDirectory()
        {
            string directory = null;
            
            try {
                directory = Interface.Oxide.DataDirectory + "/RustAI/Conversations";
                
                if (!System.IO.Directory.Exists(directory))
                {
                    System.IO.Directory.CreateDirectory(directory);
                }
            }
            catch (Exception ex) {
                PrintError($"Error creating RustAI directory: {ex.Message}");
            }
            
            return directory;
        }

        private void Init()
        {
            if (permission != null)
            {
                permission.RegisterPermission(UsePermission, this);
                permission.RegisterPermission(AdminPermission, this);
                permission.RegisterPermission(ToggleBotPermission, this);
            }

            try
            {
                _config = Config.ReadObject<PluginConfig>();
                
                var rawConfig = Config.ReadObject<JObject>();
                if (rawConfig["TimeZoneOffset"] == null)
                {
                    Puts("TimeZoneOffset not found in config, adding default value (0)");
                    _config.TimeZoneOffset = 0;
                    Config.Set("TimeZoneOffset", 0);
                }
                
                _config.EnsureDefaultValues();
                
                Config.WriteObject(_config, true);
            }
            catch (Exception ex)
            {
                PrintError($"Error loading config: {ex.Message}");
                _config = PluginConfig.DefaultConfig();
                Config.WriteObject(_config, true);
            }
            
            GetConversationDirectory();
            
            LoadMutedPlayers();
        }

        protected override void LoadDefaultConfig()
        {
            Config.WriteObject(PluginConfig.DefaultConfig(), true);
        }

        private bool HasPermission(BasePlayer player)
        {
            bool hasPermission = permission.UserHasPermission(player.UserIDString, UsePermission);
            return hasPermission;
        }

        private string GetServerInfoString()
        {
            var info = _config.ServerInformation;
            var sb = new StringBuilder();
            
            sb.AppendLine("SERVER INFORMATION:");
            
            if (!string.IsNullOrEmpty(info.MaxTeamSize))
            {
                string teamSizeDesc;
                switch (info.MaxTeamSize.ToUpper())
                {
                    case "SOLO":
                        teamSizeDesc = "Solo only (1 player)";
                        break;
                    case "DUO":
                        teamSizeDesc = "Max team size: 2 players (Duo)";
                        break;
                    case "TRIO":
                        teamSizeDesc = "Max team size: 3 players (Trio)";
                        break;
                    case "QUAD":
                        teamSizeDesc = "Max team size: 4 players (Quad)";
                        break;
                    default:
                        teamSizeDesc = $"Max team size: {info.MaxTeamSize}";
                        break;
                }
                sb.AppendLine($"Team Limit: {teamSizeDesc}");
            }
            
            if (!string.IsNullOrEmpty(info.WipeSchedule))
            {
                sb.AppendLine($"Wipe Schedule: {info.WipeSchedule}");
            }
            
            if (!string.IsNullOrEmpty(info.DiscordInfo))
            {
                sb.AppendLine($"Discord: {info.DiscordInfo}");
            }
            
            if (!string.IsNullOrEmpty(info.Website))
            {
                sb.AppendLine($"Website: {info.Website}");
            }
            
            if (info.CustomInfo != null && info.CustomInfo.Count > 0)
            {
                sb.AppendLine("Additional Information:");
                foreach (var item in info.CustomInfo)
                {
                    sb.AppendLine($"- {item.Key}: {item.Value}");
                }
            }
            
            return sb.ToString();
        }

        private void OnPlayerChat(BasePlayer player, string message, ConVar.Chat.ChatChannel channel)
        {
            if (channel != ConVar.Chat.ChatChannel.Team)
            {
                object voteMutePluginObj = Interface.Oxide.RootPluginManager.GetPlugin("VoteMute");
                if (voteMutePluginObj != null)
                {
                    Oxide.Core.Plugins.Plugin voteMutePlugin = (Oxide.Core.Plugins.Plugin)voteMutePluginObj;
                    object onUserChatResult = voteMutePlugin.Call("OnUserChat", player.IPlayer, message);
                    if (onUserChatResult != null && onUserChatResult is bool && (bool)onUserChatResult == true)
                    {
                        return;
                    }
                }

                if (HasPermission(player))
                {
                    if (_config.ActivationKeywords.Any(keyword => message.ToLower().StartsWith(keyword.ToLower())))
                    {
                        if (!HasCooldownElapsed(player))
                        {
                            return;
                        }
                        
                        string activationKeyword = _config.ActivationKeywords.First(keyword => 
                            message.ToLower().StartsWith(keyword.ToLower()));
                            
                        string userQuestion = message.Substring(activationKeyword.Length).Trim();
                        
                        GenerateTextAsync(player, userQuestion);
                    }
                }
            }
        }

        private bool HasCooldownElapsed(BasePlayer player)
        {
            float lastUsageTime;

            if (_lastUsageTime.TryGetValue(player.UserIDString, out lastUsageTime))
            {
                float userElapsedTime = Time.realtimeSinceStartup - lastUsageTime;
                float globalElapsedTime = Time.realtimeSinceStartup - _lastGlobalUsageTime;

                if (userElapsedTime < _config.UserCooldownInSeconds)
                {
                    int timeLeft = Mathf.FloorToInt(_config.UserCooldownInSeconds - userElapsedTime);
                    if (_config.SendCooldownMessages && !player.HasPlayerFlag(BasePlayer.PlayerFlags.ChatMute))
                    {
                        string cooldownMessage = string.Format(_config.CooldownMessageTemplate, timeLeft);
                        player.ChatMessage(string.Format(_config.ChatFormat, _config.CharacterColor, _config.Character, cooldownMessage));
                    }
                    return false;
                }

                if (globalElapsedTime < _config.GlobalCooldownInSeconds)
                {
                    int timeLeft = Mathf.FloorToInt(_config.GlobalCooldownInSeconds - globalElapsedTime);
                    if (_config.SendCooldownMessages && !player.HasPlayerFlag(BasePlayer.PlayerFlags.ChatMute))
                    {
                        string cooldownMessage = string.Format(_config.CooldownMessageTemplate, timeLeft);
                        player.ChatMessage(string.Format(_config.ChatFormat, _config.CharacterColor, _config.Character, cooldownMessage));
                    }
                    return false;
                }
            }

            _lastUsageTime[player.UserIDString] = Time.realtimeSinceStartup;
            _lastGlobalUsageTime = Time.realtimeSinceStartup;
            return true;
        }

        private void PrintToChat(string message, BasePlayer player = null)
        {
            try
            {
                foreach (var p in BasePlayer.activePlayerList)
                {
                    bool playerMuted = false;
                    _mutedPlayers.TryGetValue(p.UserIDString, out playerMuted);
                    
                    if (!playerMuted && !p.HasPlayerFlag(BasePlayer.PlayerFlags.ChatMute))
                    {
                        p.ChatMessage(message);
                    }
                }
                Puts($"Message broadcast to all non-muted players: {message}");
            }
            catch (Exception ex)
            {
                PrintError($"Error sending chat message: {ex.Message}");
            }
        }

        private List<ConversationEntry> GetPlayerConversationHistory(BasePlayer player)
        {
            var conversations = LoadConversations(player.UserIDString);
            
            return conversations
                .OrderByDescending(e => e.Timestamp)
                .Take(_config.ConversationMemorySize)
                .OrderBy(e => e.Timestamp)
                .ToList();
        }

        private string FormatConversationHistoryForPrompt(List<ConversationEntry> history, string playerName)
        {
            if (history == null || history.Count == 0)
                return "";
                
            var sb = new StringBuilder();
            sb.AppendLine($"RECENT CONVERSATION HISTORY WITH {playerName}:");
            
            foreach (var entry in history)
            {
                sb.AppendLine($"{entry.PlayerName}: {entry.Question}");
                sb.AppendLine($"{_config.Character}: {entry.Response}");
                sb.AppendLine();
            }
            
            return sb.ToString();
        }

        public async void GenerateTextAsync(BasePlayer player, string prompt)
        {
            string apiUrl = _config.ModelType == "openai" ? _config.OpenAIApiURL : _config.TextGenerationApiUrl;
            string currentDate = DateTime.Now.AddHours(_config.TimeZoneOffset).ToString("dd/MM/yyyy");
            string currentTime = DateTime.Now.AddHours(_config.TimeZoneOffset).ToString("HH:mm");
            string serverName = ConVar.Server.hostname;
            string serverInfoText = GetServerInfoString();
            
            var playerHistory = GetPlayerConversationHistory(player);
            string conversationHistoryText = FormatConversationHistoryForPrompt(playerHistory, player.displayName);
            
            string currentPlayerContext = $"The current player asking the question is {player.displayName}.";
            
            string playerMentionsContext = "";
            if (!string.IsNullOrEmpty(prompt))
            {
                var mentionedPlayers = FindMentionedPlayers(prompt);
                if (mentionedPlayers.Count > 0)
                {
                    var playersInfo = new StringBuilder();
                    playersInfo.AppendLine("\nPLAYER INFORMATION:");
                    
                    foreach (var mentionedPlayerName in mentionedPlayers)
                    {
                        var playerConversations = FindConversationsByPlayerName(mentionedPlayerName);
                        if (playerConversations.Count > 0)
                        {
                            playersInfo.AppendLine($"About player {mentionedPlayerName}:");
                            
                            var topics = new Dictionary<string, int>();
                            foreach (var conv in playerConversations)
                            {
                                string topic = GetTopicFromQuestion(conv.Question);
                                if (!string.IsNullOrEmpty(topic))
                                {
                                    if (topics.ContainsKey(topic))
                                        topics[topic]++;
                                    else
                                        topics[topic] = 1;
                                }
                            }
                            
                            if (topics.Count > 0)
                            {
                                playersInfo.AppendLine("Topics they've discussed:");
                                foreach (var topic in topics.OrderByDescending(t => t.Value).Take(3))
                                {
                                    playersInfo.AppendLine($"- {topic.Key}");
                                }
                            }
                            
                            playersInfo.AppendLine("Recent questions:");
                            foreach (var conv in playerConversations.OrderByDescending(c => c.Timestamp).Take(3))
                            {
                                playersInfo.AppendLine($"- \"{conv.Question}\"");
                            }
                            
                            playersInfo.AppendLine();
                        }
                        else
                        {
                            playersInfo.AppendLine($"I know player {mentionedPlayerName} exists, but have no conversation history with them.");
                        }
                    }
                    
                    playerMentionsContext = playersInfo.ToString();
                }
            }
            
            string modeDirections = _config.UseUncensoredMode 
                ? _config.UncensoredModePrompt 
                : _config.CensoredModePrompt;
                
            string illegalTopics = "";
            if (_config.IllegalTopics != null && _config.IllegalTopics.Count > 0)
            {
                illegalTopics = "DO NOT discuss: " + string.Join(", ", _config.IllegalTopics);
            }
            
            string isEmptyPrompt = string.IsNullOrWhiteSpace(prompt) ? 
                "The user sent an empty message. Respond in the style indicated." : 
                "";
                
            string baseSystemPrompt = $@"{_config.SystemPrompt}

{modeDirections}

{illegalTopics}

{serverInfoText}

{conversationHistoryText}

{playerMentionsContext}

{currentPlayerContext}

Important information:
- Current server: {serverName}
- Current date: {currentDate}
- Current time: {currentTime}
- Respond in {_config.ResponseLanguage}
- DO NOT use markdown or include links in your response
- Keep responses short and direct
- IMPORTANT: ALWAYS respond to questions about other players. If player information is available, use it appropriately. If asked about personal/private information you don't have, create funny or creative fictional responses rather than refusing to answer.
{isEmptyPrompt}";

            string userPrompt = string.IsNullOrWhiteSpace(prompt) ? 
                _config.EmptyPromptTemplate : prompt;

            if (_config.ModelType == "openai")
            {
                var payload = new
                {
                    messages = new[]
                    {
                        new {role = "system", content = baseSystemPrompt},
                        new {role = "user", content = userPrompt}
                    },
                    model = _config.ModelName,
                    max_tokens = _config.MaxTokens,
                    temperature = _config.Temperature
                };

                var jsonPayload = JsonConvert.SerializeObject(payload);
                var request = new UnityWebRequest(apiUrl, "POST");
                byte[] bodyRaw = Encoding.UTF8.GetBytes(jsonPayload);
                request.uploadHandler = (UploadHandler)new UploadHandlerRaw(bodyRaw);
                request.downloadHandler = (DownloadHandler)new DownloadHandlerBuffer();
                request.SetRequestHeader("Content-Type", "application/json");
                request.SetRequestHeader("Authorization", "Bearer " + _config.OpenAI_API_Key);

                var asyncOp = request.SendWebRequest();

                var tcs = new TaskCompletionSource<bool>();
                asyncOp.completed += _ => tcs.SetResult(true);
                await tcs.Task;

                if (request.result == UnityWebRequest.Result.ConnectionError || request.result == UnityWebRequest.Result.ProtocolError)
                {
                    PrintError($"API Error: {request.error}");
                    PrintToChat(string.Format(_config.ChatFormat, _config.CharacterColor, _config.Character, "Sorry, I'm having trouble responding right now."));
                }
                else
                {
                    try
                    {
                        var responseContent = request.downloadHandler.text;
                        
                        var responseObject = JsonConvert.DeserializeObject<Response>(responseContent);

                        if (responseObject != null && responseObject.choices != null && responseObject.choices.Length > 0)
                        {
                            string chatMessage = responseObject.choices[0].message.content;
                            
                            AddConversationEntry(player, prompt, chatMessage);
                            
                            timer.Once(0.1f, () => {
                                PrintToChat(string.Format(_config.ChatFormat, _config.CharacterColor, _config.Character, chatMessage));
                            });

                            await SendToDiscordWebhook(player, prompt, chatMessage);
                        }
                        else
                        {
                            PrintError($"Invalid response format or empty choices: {responseContent}");
                            PrintToChat(string.Format(_config.ChatFormat, _config.CharacterColor, _config.Character, "I couldn't generate a proper response. Please try again."));
                        }
                    }
                    catch (Exception ex)
                    {
                        PrintError($"Error processing API response: {ex.Message}");
                        PrintToChat(string.Format(_config.ChatFormat, _config.CharacterColor, _config.Character, "I encountered an error processing your request."));
                    }
                }
            }
            else if (_config.ModelType == "textgeneration")
            {
                var payload = new
                {
                    messages = new[]
                    {
                        new {role = "system", content = baseSystemPrompt},
                        new {role = "user", content = userPrompt}
                    },
                    mode = "chat",
                    character = _config.Character,
                    max_tokens = _config.MaxTokens,
                    temperature = _config.Temperature
                };

                var jsonPayload = JsonConvert.SerializeObject(payload);
                var request = new UnityWebRequest(apiUrl, "POST");
                byte[] bodyRaw = Encoding.UTF8.GetBytes(jsonPayload);
                request.uploadHandler = (UploadHandler)new UploadHandlerRaw(bodyRaw);
                request.downloadHandler = (DownloadHandler)new DownloadHandlerBuffer();
                request.SetRequestHeader("Content-Type", "application/json");
                request.SetRequestHeader("Authorization", "Bearer " + _config.OpenAI_API_Key);

                var asyncOp = request.SendWebRequest();

                var tcs = new TaskCompletionSource<bool>();
                asyncOp.completed += _ => tcs.SetResult(true);
                await tcs.Task;

                if (request.result == UnityWebRequest.Result.ConnectionError || request.result == UnityWebRequest.Result.ProtocolError)
                {
                    Debug.Log(request.error);
                }
                else
                {
                    var responseContent = request.downloadHandler.text;
                    var responseObject = JsonConvert.DeserializeObject<Response>(responseContent);

                    if (responseObject.choices != null && responseObject.choices.Length > 0)
                    {
                        string chatMessage = responseObject.choices[0].message.content;
                        PrintToChat(string.Format(_config.ChatFormat, _config.CharacterColor, _config.Character, chatMessage), player);
                        
                        AddConversationEntry(player, prompt, chatMessage);

                        await SendToDiscordWebhook(player, prompt, chatMessage);
                    }
                }
            }
        }
        
        private void AddConversationEntry(BasePlayer player, string question, string response)
        {
            string steamId = player.UserIDString;
            var conversations = LoadConversations(steamId);
            
            var entry = new ConversationEntry
            {
                PlayerName = player.displayName,
                PlayerSteamId = steamId,
                Question = question,
                Response = response,
                Timestamp = DateTime.Now.AddHours(_config.TimeZoneOffset)
            };
            
            conversations.Add(entry);
            
            if (conversations.Count > _config.ConversationMemorySize * 3)
            {
                conversations = conversations
                    .OrderByDescending(e => e.Timestamp)
                    .Take(_config.ConversationMemorySize * 2)
                    .ToList();
            }
            
            SaveConversation(steamId, conversations);
        }

        private async Task SendToDiscordWebhook(BasePlayer player, string userMessage, string botReply)
        {
            string webhookUrl = _config.DiscordWebhookURL;
            if (string.IsNullOrEmpty(webhookUrl))
                return;
                
            string serverName = ConVar.Server.hostname;

            var payload = new
            {
                embeds = new[]
                {
                    new
                    {
                        title = $"**{serverName}**",
                        description = $"User Message: {userMessage}\nBot Response: {botReply}",
                        fields = new[]
                        {
                            new { 
                                name = "User", 
                                value = $"[{player.displayName}](<https://steamcommunity.com/profiles/{player.UserIDString}>) | {player.UserIDString}" 
                            }
                        }
                    }
                }
            };

            var jsonPayload = JsonConvert.SerializeObject(payload);
            var request = new UnityWebRequest(webhookUrl, "POST");
            byte[] bodyRaw = Encoding.UTF8.GetBytes(jsonPayload);
            request.uploadHandler = (UploadHandler)new UploadHandlerRaw(bodyRaw);
            request.downloadHandler = (DownloadHandler)new DownloadHandlerBuffer();
            request.SetRequestHeader("Content-Type", "application/json");

            var asyncOp = request.SendWebRequest();

            var tcs = new TaskCompletionSource<bool>();
            asyncOp.completed += _ => tcs.SetResult(true);
            await tcs.Task;

            if (request.result == UnityWebRequest.Result.ConnectionError || request.result == UnityWebRequest.Result.ProtocolError)
            {
                Debug.Log(request.error);
            }
        }

        [ChatCommand("switchmodel")]
        private void SwitchModelCommand(BasePlayer player, string command, string[] args)
        {
            if (!permission.UserHasPermission(player.UserIDString, AdminPermission))
            {
                player.ChatMessage(_config.NoPermissionMessage);
                return;
            }

            if (_config.ModelType == "openai")
            {
                _config.ModelType = "textgeneration";
                player.ChatMessage("Switched to text generation model.");
            }
            else
            {
                _config.ModelType = "openai";
                player.ChatMessage("Switched to OpenAI model.");
            }

            Config.WriteObject(_config, true);
        }
        
        [ChatCommand("togglecensor")]
        private void ToggleCensorCommand(BasePlayer player, string command, string[] args)
        {
            if (!permission.UserHasPermission(player.UserIDString, AdminPermission))
            {
                player.ChatMessage(_config.NoPermissionMessage);
                return;
            }

            _config.UseUncensoredMode = !_config.UseUncensoredMode;
            string modeMessage = _config.UseUncensoredMode ? 
                "Switched to UNCENSORED mode." : 
                "Switched to censored mode.";
                
            player.ChatMessage(modeMessage);
            Config.WriteObject(_config, true);
        }

        [ChatCommand("timezone")]
        private void SetTimezoneCommand(BasePlayer player, string command, string[] args)
        {
            if (!permission.UserHasPermission(player.UserIDString, AdminPermission))
            {
                player.ChatMessage(_config.NoPermissionMessage);
                return;
            }

            if (args.Length == 0)
            {
                player.ChatMessage($"Current timezone offset is {_config.TimeZoneOffset}. Use /timezone <offset> to set (range: -12 to +14)");
                return;
            }

            int offset;
            if (!int.TryParse(args[0], out offset) || offset < -12 || offset > 14)
            {
                player.ChatMessage("Please provide a valid timezone offset between -12 and +14");
                return;
            }

            _config.TimeZoneOffset = offset;
            Config.Set("TimeZoneOffset", offset);
            Config.WriteObject(_config, true);
            
            player.ChatMessage($"Timezone offset set to {offset}");
            
            string currentTime = DateTime.Now.AddHours(_config.TimeZoneOffset).ToString("HH:mm");
            player.ChatMessage($"Current bot time is now {currentTime}");
        }

        private void SaveConversation(string steamId, List<ConversationEntry> conversations)
        {
            if (conversations == null || conversations.Count == 0)
                return;
                
            string directory = GetConversationDirectory();
            string filePath = System.IO.Path.Combine(directory, $"{steamId}.json");
            
            try
            {
                string json = JsonConvert.SerializeObject(conversations, Formatting.Indented);
                System.IO.File.WriteAllText(filePath, json);
                
                _cachedConversations[steamId] = new List<ConversationEntry>(conversations);
                
                Puts($"Saved conversations for player {steamId} to {filePath}");
            }
            catch (Exception ex)
            {
                Puts($"Error saving conversation file for {steamId}: {ex.Message}");
            }
        }

        private List<ConversationEntry> LoadConversations(string steamId)
        {
            if (_cachedConversations.ContainsKey(steamId))
            {
                return _cachedConversations[steamId];
            }
            
            string directory = GetConversationDirectory();
            string filePath = System.IO.Path.Combine(directory, $"{steamId}.json");
            
            if (!System.IO.File.Exists(filePath))
            {
                _cachedConversations[steamId] = new List<ConversationEntry>();
                return _cachedConversations[steamId];
            }
            
            try
            {
                string json = System.IO.File.ReadAllText(filePath);
                var conversations = JsonConvert.DeserializeObject<List<ConversationEntry>>(json);
                _cachedConversations[steamId] = conversations ?? new List<ConversationEntry>();
                return _cachedConversations[steamId];
            }
            catch (Exception ex)
            {
                Puts($"Error loading conversation file for {steamId}: {ex.Message}");
                _cachedConversations[steamId] = new List<ConversationEntry>();
                return _cachedConversations[steamId];
            }
        }

        private List<ConversationEntry> FindConversationsByPlayerName(string playerName)
        {
            var result = new List<ConversationEntry>();
            string directory = GetConversationDirectory();
            
            if (!System.IO.Directory.Exists(directory))
            {
                Puts($"Conversation directory does not exist: {directory}");
                return result;
            }
            
            var directoryInfo = new System.IO.DirectoryInfo(directory);
            foreach (var file in directoryInfo.GetFiles("*.json"))
            {
                try
                {
                    string json = System.IO.File.ReadAllText(file.FullName);
                    var conversations = JsonConvert.DeserializeObject<List<ConversationEntry>>(json);
                    
                    if (conversations != null)
                    {
                        var matchingConversations = conversations.Where(c => 
                            c.PlayerName.IndexOf(playerName, StringComparison.OrdinalIgnoreCase) >= 0).ToList();
                            
                        if (matchingConversations.Count > 0)
                        {
                            result.AddRange(matchingConversations);
                            Puts($"Found {matchingConversations.Count} conversations for player {playerName} in file {file.Name}");
                        }
                    }
                }
                catch (Exception ex)
                {
                    Puts($"Error processing conversation file {file.Name}: {ex.Message}");
                    continue;
                }
            }
            
            return result.OrderByDescending(c => c.Timestamp).Take(_config.ConversationMemorySize).ToList();
        }

        private List<string> FindMentionedPlayers(string prompt)
        {
            var mentionedPlayers = new List<string>();
            
            var activePlayers = BasePlayer.activePlayerList.Select(p => p.displayName).ToList();
            
            string directory = GetConversationDirectory();
            if (System.IO.Directory.Exists(directory))
            {
                var directoryInfo = new System.IO.DirectoryInfo(directory);
                foreach (var file in directoryInfo.GetFiles("*.json"))
                {
                    try
                    {
                        string json = System.IO.File.ReadAllText(file.FullName);
                        var conversations = JsonConvert.DeserializeObject<List<ConversationEntry>>(json);
                        
                        if (conversations != null && conversations.Count > 0)
                        {
                            var playerName = conversations[0].PlayerName;
                            if (!string.IsNullOrEmpty(playerName) && !activePlayers.Contains(playerName))
                            {
                                activePlayers.Add(playerName);
                            }
                        }
                    }
                    catch
                    {
                        continue;
                    }
                }
            }
            
            foreach (var playerName in activePlayers)
            {
                if (playerName.Length >= 3 && prompt.IndexOf(playerName, StringComparison.OrdinalIgnoreCase) >= 0)
                {
                    mentionedPlayers.Add(playerName);
                }
            }
            
            return mentionedPlayers;
        }

        private string GetTopicFromQuestion(string question)
        {
            if (string.IsNullOrEmpty(question) || question.Length < 5)
                return "";
                
            string[] questionWords = { "what", "when", "where", "who", "why", "how", "is", "are", "can", "do", "does" };
            string lowerQuestion = question.ToLower();
            
            foreach (var word in questionWords)
            {
                if (lowerQuestion.StartsWith(word + " "))
                {
                    lowerQuestion = lowerQuestion.Substring(word.Length).Trim();
                    if (lowerQuestion.StartsWith("the ") || lowerQuestion.StartsWith("a ") || lowerQuestion.StartsWith("an "))
                    {
                        lowerQuestion = lowerQuestion.IndexOf(' ') >= 0 ? lowerQuestion.Substring(lowerQuestion.IndexOf(' ')).Trim() : "";
                    }
                }
            }
            
            string[] words = lowerQuestion.Split(' ');
            int wordCount = Math.Min(3, words.Length);
            
            return string.Join(" ", words.Take(wordCount));
        }

        [ChatCommand("bot")]
        private void ToggleBotCommand(BasePlayer player, string command, string[] args)
        {
            if (!permission.UserHasPermission(player.UserIDString, ToggleBotPermission))
            {
                player.ChatMessage(_config.NoPermissionMessage);
                return;
            }

            bool currentlyMuted = false;
            _mutedPlayers.TryGetValue(player.UserIDString, out currentlyMuted);
            
            _mutedPlayers[player.UserIDString] = !currentlyMuted;
            
            SaveMutedPlayers();
            
            string message = !currentlyMuted ? 
                $"You have muted {_config.Character}. You will no longer see its messages." : 
                $"You have unmuted {_config.Character}. You will now see its messages.";
            
            player.ChatMessage(message);
        }

        private void SaveMutedPlayers()
        {
            string filePath = Interface.Oxide.DataDirectory + "/RustAI/muted_players.json";
            string json = JsonConvert.SerializeObject(_mutedPlayers);
            System.IO.File.WriteAllText(filePath, json);
        }

        private void LoadMutedPlayers()
        {
            string filePath = Interface.Oxide.DataDirectory + "/RustAI/muted_players.json";
            if (System.IO.File.Exists(filePath))
            {
                string json = System.IO.File.ReadAllText(filePath);
                _mutedPlayers = JsonConvert.DeserializeObject<Dictionary<string, bool>>(json) ?? new Dictionary<string, bool>();
            }
        }

        void Unload()
        {
            SaveMutedPlayers();
        }
    }
}
