/*▄▄▄▄    ███▄ ▄███▓  ▄████  ▄▄▄██▀▀▀▓█████▄▄▄█████▓
 ▓█████▄ ▓██▒▀█▀ ██▒ ██▒ ▀█▒   ▒██   ▓█   ▀▓  ██▒ ▓▒
 ▒██▒ ▄██▓██    ▓██░▒██░▄▄▄░   ░██   ▒███  ▒ ▓██░ ▒░
 ▒██░█▀  ▒██    ▒██ ░▓█  ██▓▓██▄██▓  ▒▓█  ▄░ ▓██▓ ░ 
 ░▓█  ▀█▓▒██▒   ░██▒░▒▓███▀▒ ▓███▒   ░▒████▒ ▒██▒ ░ 
 ░▒▓███▀▒░ ▒░   ░  ░ ░▒   ▒  ▒▓▒▒░   ░░ ▒░ ░ ▒ ░░   
 ▒░▒   ░ ░  ░      ░  ░   ░  ▒ ░▒░    ░ ░  ░   ░    
  ░    ░ ░      ░   ░ ░   ░  ░ ░ ░      ░    ░      
  ░             ░         ░  ░   ░      ░  ░*/
using Network;
using Newtonsoft.Json;
using System.Collections.Generic;
using System.Linq;
using UnityEngine;
namespace Oxide.Plugins
{
    [Info("ShowBB", "bmgjet", "1.0.0")]
    [Description("Shows Building Block Spheres")]
    class ShowBB : RustPlugin
    {
        //Vars
        private static ShowBB plugin;
        private Dictionary<SphereEntity, List<BasePlayer>> Spheres = new Dictionary<SphereEntity, List<BasePlayer>>();
        private readonly string RedZone = "assets/bundled/prefabs/modding/events/twitch/br_sphere_red.prefab";

        #region Configuration
        private Configuration config;
        private class Configuration
        {
            [JsonProperty("Extra Distance From Radius To Extend View")]
            public int CullOffset = 20;

            [JsonProperty("Show Safe Zones With Safe Zone Sphere")]
            public bool ShowSafeZone = false;

            [JsonProperty("Safe Zone Sphere")]
            public string SafeZoneSphere = "assets/bundled/prefabs/modding/events/twitch/br_sphere_purple.prefab";

            [JsonProperty("Component Tick Rate (sec)")]
            public float TickRate = 3f;

            public string ToJson() => JsonConvert.SerializeObject(this);

            public Dictionary<string, object> ToDictionary() => JsonConvert.DeserializeObject<Dictionary<string, object>>(ToJson());
        }

        protected override void LoadDefaultConfig() { config = new Configuration(); }
        protected override void LoadConfig()
        {
            base.LoadConfig();
            try
            {
                config = Config.ReadObject<Configuration>();
                if (config == null) { throw new JsonException(); }

                if (!config.ToDictionary().Keys.SequenceEqual(Config.ToDictionary(x => x.Key, x => x.Value).Keys))
                {
                    PrintWarning("Configuration appears to be outdated; updating and saving");
                    SaveConfig();
                }
            }
            catch
            {
                PrintWarning($"Configuration file {Name}.json is invalid; using defaults");
                LoadDefaultConfig();
            }
        }
        protected override void SaveConfig()
        {
            PrintWarning($"Configuration changes saved to {Name}.json");
            Config.WriteObject(config, true);
        }
        #endregion Configuration

        #region Oxide Hooks
        private void OnServerInitialized()
        {
            plugin = this;
            //Scan Map File
            foreach (var pd in World.Serialization.world.prefabs)
            {
                if (pd.id == 3224970585) //Prevent Building Sphere In Map File
                {
                    //Create Red Sphere
                    CreateSphere(new Vector3(pd.position.x, pd.position.y, pd.position.z), pd.scale.x, RedZone, 2);
                    continue;
                }
                else if (config.ShowSafeZone && pd.id == 316558065) //Safe Zone Sphere In Map File
                {
                    //Create Safe Zone Sphere
                    CreateSphere(new Vector3(pd.position.x, pd.position.y, pd.position.z), pd.scale.x * 2, config.SafeZoneSphere, 0);
                    continue;
                }
            }
            Puts("Found " + Spheres.Count + " Spheres");
            //Check online player held entity
            timer.Once(2f, () => { foreach (var player in BasePlayer.activePlayerList) { CheckValidHeldItem(player); } });
        }

        private void Unload()
        {
            //Remove component from players.
            foreach (var p in BasePlayer.activePlayerList)
            {
                ShowBBController BBC = p.GetComponent<ShowBBController>();
                if (!BBC) { GameObject.Destroy(BBC); }
            }
            foreach (var s in Spheres.Keys)
            {
                //Destroy Sphere
                s?.Kill();
            }
            plugin = null;
        }

        private void OnActiveItemChanged(BasePlayer player, Item oldItem, Item newItem)
        {
            if (newItem != null) //Check has item and is a actual player
            {
                CheckValidHeldItem(player);
            }
        }
        #endregion

        #region Methods
        private void CheckValidHeldItem(BasePlayer player)
        {
            if (player.UserIDString.IsSteamId()) //Make sure valid player and not npc
            {
                Item i = player?.GetActiveItem(); //Check held item
                if (i?.info?.itemid != 1525520776 && !i?.info?.GetComponent<ItemModDeployable>()) { return; } //Not valid deployable or planner
                if (player.GetComponent<ShowBBController>() == null) { player.gameObject.AddComponent<ShowBBController>().player = player; } //Add component if player doesn't have it
            }
        }

        private void CreateSphere(Vector3 pos, float scale, string type, int offset)
        {
            //Create sphere entity and limit networking then spawn and resize
            var sphere = GameManager.server.CreateEntity(type, pos) as SphereEntity;
            if (sphere.net == null) { sphere.net = Network.Net.sv.CreateNetworkable(); }
            Spheres.Add(sphere, new List<BasePlayer>());
            sphere._limitedNetworking = true;
            sphere.Spawn();
            sphere.LerpRadiusTo(scale + offset, scale);
        }

        private void HideSphere(SphereEntity s, BasePlayer player)
        {
            //Send packet to destroy it on clients screen with no gibs.
            NetWrite netWrite = Network.Net.sv.StartWrite();
            netWrite.PacketID(Message.Type.EntityDestroy);
            netWrite.EntityID(s.net.ID);
            netWrite.UInt8((byte)BaseNetworkable.DestroyMode.None);
            netWrite.Send(new SendInfo(player.Connection));
            Spheres[s].Remove(player); //Remove from snapshot list
        }

        private void ShowSphere(SphereEntity s, BasePlayer player)
        {
            if (!Spheres[s].Contains(player)) //Make sure sphere hasnt already sent snapshot to that player
            {
                //Send Snapshot to player of sphere to redraw it on screen
                Spheres[s].Add(player);
                s.SendAsSnapshot(player.Connection, false);
            }
        }
        #endregion

        #region Class
        public class ShowBBController : MonoBehaviour
        {
            public BasePlayer player;
            private float check;
            private float tick;
            private Item item;

            public void OnDestroy()
            {
                if (player && plugin)
                {
                    //Clean Up
                    for (int i = plugin.Spheres.Count - 1; i >= 0; i--)
                    {
                        var S = plugin.Spheres.ElementAt(i);
                        if (S.Key == null || S.Value == null) { continue; }
                        if (S.Value.Contains(player)) { plugin.HideSphere(S.Key, player); }
                    }
                }
            }

            public void FixedUpdate()
            {
                if (check > Time.realtimeSinceStartup) { return; }
                check = (Time.realtimeSinceStartup + 0.25f); //250ms tick
                if (!player || !plugin)  //Destroy if no player or plugin
                {
                    Destroy(this);
                    return;
                }
                Item item2 = player?.GetActiveItem();
                if (item != item2) //Only recheck if active item changed.
                {
                    if (item2?.info?.itemid != 1525520776 && !item2?.info?.GetComponent<ItemModDeployable>())
                    {
                        //Condition lost destroy component
                        Destroy(this);
                        return;
                    }
                    item = item2; //Validated Item
                }
                //Check networking
                if (tick > Time.realtimeSinceStartup) { return; }
                tick = (Time.realtimeSinceStartup + plugin.config.TickRate); //Distance/Draw Tick
                bool dead = false;
                for (int i = plugin.Spheres.Count - 1; i >= 0; i--)
                {
                    var S = plugin.Spheres.ElementAt(i);
                    if (S.Key == null) { dead = true; continue; }
                    dead = false;
                    if (player.Distance2D(S.Key.transform.position) < (S.Key.lerpRadius / 2) + plugin.config.CullOffset)
                    {
                        //within distance
                        plugin.ShowSphere(S.Key, player);
                        continue;
                    }
                    if (S.Value.Contains(player) && player.Distance2D(S.Key.transform.position) > (S.Key.lerpRadius / 2) + plugin.config.CullOffset)
                    {
                        //out of distance
                        plugin.HideSphere(S.Key, player);
                    }
                }
                if (dead) { Destroy(this); }
            }
        }
        #endregion
    }
}