using System;
using System.Collections.Generic;
using Oxide.Core.Plugins;
using System.Linq;
using Newtonsoft.Json;

namespace Oxide.Plugins
{
    [Info("Solo Notifier", "crunch", "1.0.1")]
    [Description("Notifies raider if a base is owned by a solo")]
    public class SoloNotifier : RustPlugin
    {

        [PluginReference] Plugin UINotify, Notify;

        private Configuration _pluginConfig;
        List<object> prefabs;
        public bool soloTeam;


        #region Main
        object OnEntityTakeDamage(BaseCombatEntity entity, HitInfo info)
        {
            if (entity == null
                || entity is BasePlayer
                || !entity.OwnerID.IsSteamId()
                || info?.InitiatorPlayer == null
                || info.InitiatorPlayer.IsNpc)
            {
                return null;
            }

            if (IsBuilding(entity))
            {
                CheckStructure(entity, info);
            }
            return null;
        }

        public void CheckStructure(BaseEntity entity, HitInfo info)
        {
            soloTeam = false;

            ulong targetID = entity.OwnerID;
            var attacker = info.InitiatorPlayer;

            if (targetID == 0 || attacker == null) return;

            if (attacker.userID == targetID) return;

            BuildingPrivlidge privilege = entity.GetBuildingPrivilege();

            if (privilege == null) return;

            if (IsDecaying(privilege))
                return;

            if (targetID.IsSteamId() && !soloNotify.ContainsKey(attacker.userID))
            {
                var authlist = privilege.authorizedPlayers.Select(x => x).ToList();

                foreach (var auth in authlist)
                {
                    if (auth == attacker.userID)
                    {
                        AddNotifyTimer(attacker.userID, attacker.displayName);
                        return;
                    }
                }
                var player = BasePlayer.FindAwakeOrSleeping(Convert.ToString(targetID));
                if (player == null)
                {
                    soloTeam = true;
                }

                if (player != null && player.currentTeam != 0UL)
                {
                    var teamMembers = player.Team.members.ToList();
                    var getTeam = $"{string.Join(", ", teamMembers)}";

                    if (teamMembers.Count < 2) soloTeam = true;
                }

                if (player != null && player.currentTeam == 0)
                {
                    soloTeam = true;
                }

                var getAuth = $"{string.Join(", ", authlist)}";
                if (authlist.Count < 2 && soloTeam)
                {
                    if (UINotify)
                    {
                        UINotify?.Call("SendNotify", attacker.userID, _pluginConfig.uiType, _pluginConfig.notifyMessage);
                    }
                    else if (Notify)
                    {
                        Notify?.Call("SendNotify", attacker.userID, _pluginConfig.uiType, _pluginConfig.notifyMessage);
                    }
                    else
                    {
                        attacker.ChatMessage($"{_pluginConfig.notifyMessage}");
                    }

                    if (_pluginConfig.notifyLog) Puts($"{attacker.displayName} notified of solo base");
                }

                AddNotifyTimer(attacker.userID, attacker.displayName);
            }

            return;
        }

        private bool IsDecaying(BuildingPrivlidge privilege) =>
            privilege == null || privilege.GetProtectedMinutes(true) <= 0;

        public Dictionary<string, bool> holdBlock = new Dictionary<string, bool>();

        public bool IsBuilding(BaseCombatEntity entity)
        {
            if (entity is BuildingBlock)
                return true;

            var result = false;

            if (!result && !string.IsNullOrEmpty(entity.ShortPrefabName))
            {
                var prefabName = entity.ShortPrefabName;

                if (holdBlock.TryGetValue(prefabName, out result))
                    return result;

                foreach (string p in _pluginConfig.DefaultPrefabs)
                    if (prefabName.IndexOf(p) != -1)
                    {
                        result = true;
                        break;
                    }

                if (!holdBlock.ContainsKey(prefabName))
                    holdBlock.Add(prefabName, result);
            }

            return result;
        }
        #endregion

        #region NotifyTimer
        Dictionary<ulong, Timer> soloNotify = new Dictionary<ulong, Timer>();

        void AddNotifyTimer(ulong playerID, string playerName)
        {
            if (!soloNotify.ContainsKey(playerID))
            {
                soloNotify.Add(playerID, NotifyTimer(playerID, playerName));
            }
        }

        void RemoveNotifyTimer(ulong playerID, string playerName)
        {
            if (soloNotify.ContainsKey(playerID))
            {
                soloNotify[playerID].Destroy();
                soloNotify.Remove(playerID);
            }
        }

        Timer NotifyTimer(ulong playerID, string playerName)
        {
            return timer.Once(_pluginConfig.Cooldown, () =>
            {
                RemoveNotifyTimer(playerID, playerName);
            });
        }
        #endregion


        #region Configuration
        private class Configuration
        {
            [JsonProperty("Cooldown time after attack")]
            public int Cooldown = 300;

            [JsonProperty("Message to attacker")]
            public string notifyMessage = "This player is a solo";

            [JsonProperty("Log to console when player notified")]
            public bool notifyLog = true;

            [JsonProperty("UINotify Type (see https://umod.org/plugins/ui-notify)")]
            public int uiType = 6;

            [JsonProperty("Prefabs to check")]
            public List<object> DefaultPrefabs = new List<object>
            {
                    "door.hinged",
                    "door.double.hinged",
                    "floor.frame",
                    "wall.frame",
                    "wall.external",
                    "gates.external",
                    "window.bars",
                    "shutter",
                    "floor.ladder.hatch"
            };
        }

        private Configuration GetDefaultConfig() => new Configuration();

        protected override void LoadDefaultConfig() => _pluginConfig = GetDefaultConfig();

        protected override void LoadConfig()
        {
            base.LoadConfig();
            try
            {
                _pluginConfig = Config.ReadObject<Configuration>();
                if (_pluginConfig == null)
                {
                    throw new JsonException();
                }

            }
            catch (Exception e)
            {
                Puts($"{e.Message}");
                Puts($"Configuration file {Name}.json is invalid; using defaults");
                LoadDefaultConfig();
            }
        }
        protected override void SaveConfig()
        {
            Puts($"Configuration changes saved to {Name}.json");
            Config.WriteObject(_pluginConfig, true);
        }
        #endregion
    }
}