/*
This plugin (the software) is © copyright Von.

You may not copy, modify, merge, publish, distribute, sublicense, or sell copies of this software without explicit consent from the_kiiiing.

DISCLAIMER:

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.

*/

using System.Collections.Generic;
using System;
using UnityEngine;
using Oxide.Core;
using Oxide.Game.Rust.Cui;
using Network;
using System.Linq;
using Oxide.Core.Libraries.Covalence;

namespace Oxide.Plugins
{
    [Info("Timers", "Von", "1.0.0")]
    [Description("Just a simple timer plugin")]
    class Timers : RustPlugin
    {
        private const string PERMISSION_USE = "timers.use";
        private string timerUI = "";
        private DateTime? endTime = null;
        private string timerText = "";
        private HashSet<BasePlayer> playersWithUI = new HashSet<BasePlayer>();
        private Configuration config;

        class Configuration
        {
            public string BackgroundColor = "#CC000000";
            public string TextColor = "#FFFFFF";
            public bool EnableRestartTimer = true;
            public bool TransparentBackground = false;
            public int FontSize = 14;
        }

        protected override void LoadConfig()
        {
            base.LoadConfig();
            try
            {
                config = Config.ReadObject<Configuration>();
                if (config == null) throw new Exception();
            }
            catch
            {
                LoadDefaultConfig();
            }
        }

        protected override void LoadDefaultConfig()
        {
            config = new Configuration();
            SaveConfig();
        }

        protected override void SaveConfig() => Config.WriteObject(config);

        void Init()
        {
            permission.RegisterPermission(PERMISSION_USE, this);
            cmd.AddConsoleCommand("timer", this, nameof(RconTimer));
            cmd.AddConsoleCommand("timer.start", this, nameof(ConsoleStart));
            cmd.AddConsoleCommand("timer.stop", this, nameof(ConsoleStop));
        }

        void OnServerInitialized()
        {
            cmd.AddChatCommand("timer", this, "CmdTimer");
        }

        void OnServerShutdown()
        {
            if (config.EnableRestartTimer && !endTime.HasValue)
            {
                StartTimer(300, "Server Restarting");
            }
        }

        void Unload()
        {
            foreach (BasePlayer player in BasePlayer.activePlayerList)
            {
                DestroyUI(player);
            }
        }

        void OnServerCommand(ConsoleSystem.Arg arg)
        {
            if (!config.EnableRestartTimer) return;
            
            string command = arg.cmd?.Name?.ToLower();
            if (command == "restart" && arg.HasArgs(1))
            {
                if (int.TryParse(arg.Args[0], out int seconds))
                {
                    StartTimer(seconds, "Server Restarting");
                }
            }
        }

        void ConsoleStart(ConsoleSystem.Arg arg)
        {
            if (!arg.HasArgs(2))
            {
                arg.ReplyWith("Usage: timer.start <seconds> <text>");
                return;
            }

            if (!int.TryParse(arg.Args[0], out int seconds))
            {
                arg.ReplyWith("Please provide a valid number of seconds!");
                return;
            }

            string text = string.Join(" ", arg.Args.Skip(1));
            if (text.Length > 30)
            {
                arg.ReplyWith("Timer text cannot exceed 30 characters!");
                return;
            }

            timerText = text;
            StartTimer(seconds, text);
            arg.ReplyWith("Timer started!");
        }

        void ConsoleStop(ConsoleSystem.Arg arg)
        {
            if (!endTime.HasValue)
            {
                arg.ReplyWith("No timer is currently running!");
                return;
            }

            endTime = null;
            timerText = "";
            foreach (BasePlayer p in BasePlayer.activePlayerList)
            {
                DestroyUI(p);
            }
            arg.ReplyWith("Timer stopped!");
        }

        void RconTimer(ConsoleSystem.Arg arg)
        {
            if (!arg.HasArgs())
            {
                arg.ReplyWith("Usage: timer start <seconds> <text> or timer stop");
                return;
            }

            switch (arg.Args[0].ToLower())
            {
                case "start":
                    if (!arg.HasArgs(2))
                    {
                        arg.ReplyWith("Usage: timer start <seconds> <text>");
                        return;
                    }

                    if (!int.TryParse(arg.Args[1], out int seconds))
                    {
                        arg.ReplyWith("Please provide a valid number of seconds!");
                        return;
                    }

                    string text = arg.Args.Length > 2 ? string.Join(" ", arg.Args.Skip(2)) : "";
                    if (text.Length > 30)
                    {
                        arg.ReplyWith("Timer text cannot exceed 30 characters!");
                        return;
                    }
                    timerText = text;
                    StartTimer(seconds, text);
                    arg.ReplyWith("Timer started!");
                    break;

                case "stop":
                    if (endTime.HasValue)
                    {
                        endTime = null;
                        timerText = "";
                        foreach (BasePlayer p in BasePlayer.activePlayerList)
                        {
                            DestroyUI(p);
                        }
                        arg.ReplyWith("Timer stopped!");
                    }
                    else
                    {
                        arg.ReplyWith("No timer is currently running!");
                    }
                    break;

                default:
                    arg.ReplyWith("Usage: timer start <seconds> <text> or timer stop");
                    break;
            }
        }

        void CmdTimer(BasePlayer player, string command, string[] args)
        {
            if (!permission.UserHasPermission(player.UserIDString, PERMISSION_USE))
            {
                SendReply(player, "You don't have permission to use this command!");
                return;
            }

            if (args.Length == 0)
            {
                SendReply(player, "Usage: /timer start <seconds> <text> or /timer stop");
                return;
            }

            switch (args[0].ToLower())
            {
                case "start":
                    if (args.Length < 2)
                    {
                        SendReply(player, "Usage: /timer start <seconds> <text>");
                        return;
                    }

                    if (!int.TryParse(args[1], out int seconds))
                    {
                        SendReply(player, "Please provide a valid number of seconds!");
                        return;
                    }

                    string text = args.Length > 2 ? string.Join(" ", args.Skip(2)) : "";
                    if (text.Length > 30)
                    {
                        SendReply(player, "Timer text cannot exceed 30 characters!");
                        return;
                    }
                    timerText = text;
                    StartTimer(seconds, text);
                    break;

                case "stop":
                    if (endTime.HasValue)
                    {
                        endTime = null;
                        timerText = "";
                        foreach (BasePlayer p in BasePlayer.activePlayerList)
                        {
                            DestroyUI(p);
                        }
                        SendReply(player, "Timer stopped!");
                    }
                    else
                    {
                        SendReply(player, "No timer is currently running!");
                    }
                    break;

                default:
                    SendReply(player, "Usage: /timer start <seconds> <text> or /timer stop");
                    break;
            }
        }

        void StartTimer(int seconds, string text = "")
        {
            endTime = DateTime.Now.AddSeconds(seconds);
            timerText = text;
            
            foreach (BasePlayer player in BasePlayer.activePlayerList)
            {
                CreateUI(player);
            }

            timer.Every(1f, () => {
                if (!endTime.HasValue) return;

                TimeSpan remaining = endTime.Value - DateTime.Now;
                if (remaining.TotalSeconds <= 0)
                {
                    endTime = null;
                    timerText = "";
                    foreach (BasePlayer player in BasePlayer.activePlayerList)
                    {
                        DestroyUI(player);
                    }
                    return;
                }

                UpdateAllUI();
            });
        }

        void OnPlayerConnected(BasePlayer player)
        {
            if (endTime.HasValue)
            {
                CreateUI(player);
            }
        }

        void OnPlayerDisconnected(BasePlayer player)
        {
            DestroyUI(player);
            playersWithUI.Remove(player);
        }

        void CreateUI(BasePlayer player)
        {
            DestroyUI(player);

            string displayText = GetTimeString();
            float charWidth = 3f;
            float textWidth = displayText.Length * charWidth;
            float padding = 10f;
            float totalWidth = textWidth + (padding * 2);
            float minWidth = 40f;
            float width = Math.Max(totalWidth, minWidth) / Screen.width;
            string containerName = "Timer_Container";
            
            var elements = new CuiElementContainer();

            elements.Add(new CuiElement
            {
                Parent = "Hud",
                Components = {
                    new CuiImageComponent { 
                        Color = config.TransparentBackground ? "0 0 0 0" : HexToRustColor(config.BackgroundColor)
                    },
                    new CuiRectTransformComponent {
                        AnchorMin = $"{1 - width} 0.95",
                        AnchorMax = "1 0.98"
                    }
                },
                Name = containerName
            });

            elements.Add(new CuiElement
            {
                Parent = containerName,
                Components = {
                    new CuiTextComponent { 
                        Text = displayText,
                        FontSize = config.FontSize,
                        Align = TextAnchor.MiddleCenter,
                        Color = HexToRustColor(config.TextColor)
                    },
                    new CuiRectTransformComponent {
                        AnchorMin = "0 0",
                        AnchorMax = "1 1"
                    }
                },
                Name = "Timer_Text"
            });

            CuiHelper.AddUi(player, elements);
            playersWithUI.Add(player);
        }

        void UpdateAllUI()
        {
            foreach (BasePlayer player in playersWithUI)
            {
                if (player == null || !player.IsConnected) continue;
                
                CuiHelper.DestroyUi(player, "Timer_Container");
                CreateUI(player);
            }
        }

        void DestroyUI(BasePlayer player)
        {
            CuiHelper.DestroyUi(player, "Timer_Container");
        }

        string GetTimeString()
        {
            if (!endTime.HasValue) return "";
            
            TimeSpan remaining = endTime.Value - DateTime.Now;
            string timeStr;

            if (remaining.TotalMinutes >= 60)
            {
                timeStr = $"{(int)remaining.TotalHours}h {remaining.Minutes}m";
            }
            else if (remaining.TotalSeconds >= 60)
            {
                timeStr = $"{remaining.Minutes}m {remaining.Seconds}s";
            }
            else
            {
                timeStr = $"{remaining.Seconds}s";
            }

            return string.IsNullOrEmpty(timerText) ? timeStr : $"{timerText}: {timeStr}";
        }

        string HexToRustColor(string hex)
        {
            if (string.IsNullOrEmpty(hex)) return "1 1 1 1";
            
            try
            {
                hex = hex.TrimStart('#');
                if (hex.Length == 8)
                {
                    byte a = Convert.ToByte(hex.Substring(0, 2), 16);
                    byte r = Convert.ToByte(hex.Substring(2, 2), 16);
                    byte g = Convert.ToByte(hex.Substring(4, 2), 16);
                    byte b = Convert.ToByte(hex.Substring(6, 2), 16);
                    return $"{r/255f} {g/255f} {b/255f} {a/255f}";
                }
                else if (hex.Length == 6)
                {
                    byte r = Convert.ToByte(hex.Substring(0, 2), 16);
                    byte g = Convert.ToByte(hex.Substring(2, 2), 16);
                    byte b = Convert.ToByte(hex.Substring(4, 2), 16);
                    return $"{r/255f} {g/255f} {b/255f} 1";
                }
            }
            catch
            {
                return "1 1 1 1";
            }
            
            return "1 1 1 1";
        }
    }
}
