﻿using ProtoBuf;
using System.Collections.Generic;
using UnityEngine;
using Facepunch;
using System.Text;

namespace Oxide.Plugins
{
    [Info("TugAuth", "SiCkNeSs", "1.0.3")]
    [Description("Get the list of authorized players on a tugboat")]
    public class TugAuth : RustPlugin
    {
        string usePerm = "tugauth.use";


        private void OnServerInitialized()
        {
            permission.RegisterPermission(usePerm, this);
        }


        [ConsoleCommand("tugauth")]
        private void tugAuthConsole(ConsoleSystem.Arg arg)
        {
            var player = arg.Player();

            if (!arg.IsRcon && (player == null || !permission.UserHasPermission(player.UserIDString, usePerm)))
            {
                arg.ReplyWith("You do not have permission to use this command!");
                return;
            }

            if (!getSearchPos(arg, player, out Vector3 searchPosition))
            {
                arg.ReplyWith("Invalid arguments. Provide valid float values for X, Y, and Z, or rely on the player's current position.");
                return;
            }

            if (!arg.IsRcon)
            {
                tugCommand(searchPosition, true, player);
                return;
            }

            tugCommand(searchPosition);
        }


        private bool getSearchPos(ConsoleSystem.Arg arg, BasePlayer player, out Vector3 position)
        {
            position = Vector3.zero;

            if (arg.Args.Length == 3)
            {
                if (arg.Args.Length == 3 &&
                    float.TryParse(arg.Args[0], out float x) &&
                    float.TryParse(arg.Args[1], out float y) &&
                    float.TryParse(arg.Args[2], out float z))
                {
                    position = new Vector3(x, y, z);

                    return true;
                }

                return false;
            }

            if (player != null)
            {
                position = player.transform.position;
                return true;
            }

            return false;
        }


        [ChatCommand("tugauth")]
        private void tugAuthChat(BasePlayer player)
        {
            if (!permission.UserHasPermission(player.UserIDString, usePerm))
            {
                SendReply(player, "You do not have permission to use this command!");
                return;
            }

            SendReply(player, "Check the console!");

            tugCommand(player.transform.position, true, player);
        }


        private void tugCommand(Vector3 position, bool playerCommand = false, BasePlayer player = null)
        {
            var closestTug = GetClosestTug(position);

            if (closestTug == null)
            {
                SendMessage("No TugBoat found near the specified position.", playerCommand, player);
                return;
            }

            var authList = ListAuthed(closestTug);
            if (authList == null || authList.Count == 0)
            {
                SendMessage("There are no authorized users on this TugBoat.", playerCommand, player);

                return;
            }

            var sb = Pool.Get<StringBuilder>();
            try
            {
                sb.Append("Listing Tug Authorized Players:");
                foreach (var auth in authList)
                {
                    if (auth != null)
                    {
                        sb.AppendLine($"{auth.username} - {auth.userid}");
                    }
                }

                SendMessage(sb.ToString(), playerCommand, player);
            }
            finally
            {
                Pool.FreeUnmanaged(ref sb);
            }
        }


        private void SendMessage(string message, bool playerCommand = false, BasePlayer player = null)
        {
            if (playerCommand)
            {
                player.ConsoleMessage(message);
            }
            else
            {
                Puts(message);
            }
        }


        private List<PlayerNameID> ListAuthed(BaseVehicle tugBoat)
        {
            foreach (BaseEntity child in tugBoat.children)
            {
                VehiclePrivilege vehiclePrivilege = child as VehiclePrivilege;
                if (!(vehiclePrivilege == null))
                {
                    return vehiclePrivilege.authorizedPlayers;
                }
            }

            return new List<PlayerNameID>();
        }


        private Tugboat GetClosestTug(Vector3 pos)
        {
            var closest = -1f;
            Tugboat result = null;

            List<Tugboat> tugboats = Pool.Get<List<Tugboat>>();
            try
            {
                foreach (BaseNetworkable entity in BaseNetworkable.serverEntities)
                {
                    if(entity is Tugboat tugboat){
                        tugboats.Add(tugboat);
                    }
                }

                for (int i = 0; i < tugboats.Count; i++)
                {
                    var dist = Vector3.Distance(pos, tugboats[i].transform.position);

                    if (closest < 0 || dist < closest)
                    {
                        closest = dist;
                        result = tugboats[i];
                    }
                }

            }
            finally
            {
                Pool.FreeUnmanaged(ref tugboats);
            }

            return result;
        }
    }
}