using Newtonsoft.Json;
using System.Collections.Generic;
using System.Linq;
using UnityEngine;
using Random = UnityEngine.Random;

namespace Oxide.Plugins
{
    [Info("Workcart Spawner", "SPooCK", "1.3.1")]
    [Description("Auto monitor and replace Workcarts for custom maps.")]
    class WorkcartSpawner : RustPlugin
    {
        #region Config
        class CuPrefabs {
            [JsonProperty("Spawns in random order (prefab, customise)")]
            public Dictionary<string, CustomWK> Prefabs = new Dictionary<string, CustomWK>() {
                { "assets/content/vehicles/trains/workcart/workcart_aboveground.entity.prefab", new CustomWK() },
                { "assets/content/vehicles/trains/workcart/workcart_aboveground2.entity.prefab", new CustomWK() }
            };
        }
        
        class CustomWK {
            [JsonProperty("Enabled")]
            public bool Enabled = false;

            [JsonProperty("Max Health (default 1000)")]
            public float maxHealth = -1;

            [JsonProperty("Engine Force (default 30000)")]
            public float engineForce = -1;

            [JsonProperty("Maximum Speed (default 18)")]
            public float maxSpeed = -1;

            [JsonProperty("Fuel Storage Store All Items (max slots must be > 1)")]
            public bool Store = false;

            [JsonProperty("Fuel Storage Maximum Slots (default 1)")]
            public int Slots = -1;

            [JsonProperty("Max Fuel per second (default 0.075)")]
            public float maxFuelPerSec = -1;

            [JsonProperty("Idle Fuel per second (default 0.025)")]
            public float idleFuelPerSec = -1;

            [JsonProperty("Driver Protection Density (default 1) (Range 0-100)")]
            public float protDensity = -1;

            [JsonProperty("Engine Startup Time (default 0.25)")]
            public float engineStartupTime = -1;

            [JsonProperty("Engine Damage to Slow (default 200)")]
            public float engineDamageToSlow = -1;

            [JsonProperty("Engine Damage Time Frame (default 10)")]
            public float engineDamageTimeframe = -1;

            [JsonProperty("Engine Slow Time (default 8)")]
            public float engineSlowedTime = -1;

            [JsonProperty("Engine Slowed max Velocity (default 4)")]
            public float engineSlowedMaxVel = -1;
        }

        private Configuration Settings;

        private class Configuration {
            [JsonProperty("Workcart Prefabs (-1 = unchanged)")]
            public CuPrefabs WKPrefabs = new CuPrefabs();

            [JsonProperty("Find Prefabs Name")]
            public List<string> Prefabs;

            [JsonProperty("Using Custom Fuel Inventory (if not used, leave false)")]
            public bool CustomFuel = false;

            [JsonProperty("Terrain Passing Collision Enabled")]
            public bool TerrColl = true;

            [JsonProperty("Default spawn distance from the Prefab")]
            public float Distance  = 5f;

            [JsonProperty("Time to respawn after Death (seconds)")]
            public int Time;

            public static Configuration Generate() {
                return new Configuration {
                    Prefabs = new List<string>() {
                        "assets/content/structures/train_tracks/train_track_3x3_end.prefab"
                    }
                };
            }
        }

        protected override void SaveConfig() => Config.WriteObject(Settings);

        protected override void LoadDefaultConfig() {
            Settings = Configuration.Generate();
            SaveConfig();
        }

        protected override void LoadConfig() {
            base.LoadConfig();
            try {
                Settings = Config.ReadObject<Configuration>();
                if (Settings?.WKPrefabs == null) LoadDefaultConfig();
                SaveConfig();
            } catch {
                PrintError("Error reading config, please check !");
            }

            if (Settings.Time < 1) {
                Settings.Time = 10;
                SaveConfig();
                PrintError("Respawn Time under 1 sec. ! Reverted to 10.");
            }
        }
        #endregion

        #region Data
        static readonly int LowGradeFuel = -946369541;
        static int passLayer = ToLayer(LayerMask.GetMask("Trigger")); // 18
        private readonly Dictionary<GameObject, Timer> Timers = new Dictionary<GameObject, Timer>();
        private Dictionary<GameObject, TrainEngine> TrackObjects = new Dictionary<GameObject, TrainEngine>();

        void Unload() {
            foreach (KeyValuePair<GameObject, TrainEngine> entry in TrackObjects) 
                if (entry.Value != null) entry.Value.Kill();
            TrackObjects.Clear();

            foreach (KeyValuePair<GameObject, Timer> entry in Timers) entry.Value.Destroy();
            Timers.Clear();
        }

        void OnServerInitialized() {
            SpawnAllWorkCarts();
        }
        #endregion

        #region Methods
        private void SpawnAllWorkCarts() {
            if (Settings.WKPrefabs.Prefabs.Count == 0) {
                PrintError("No Workcart Prefabs !");
                return;
            }

            foreach (GameObject Track in UnityEngine.Object.FindObjectsOfType<GameObject>().Where(o => Settings.Prefabs.Contains(o.name)))
                RespawnWorkCart(Track);
        }

        private void SetupCustomisation(TrainEngine trainEngine, CustomWK caConfig) {
            //string info = $"HP: [{trainEngine.MaxHealth()}] EForce: [{trainEngine.engineForce}] MSpeed: [{trainEngine.maxSpeed}] MFuPerSec: [{trainEngine.maxFuelPerSec}]" +
            //    $" IdleFuPerSec: [{trainEngine.idleFuelPerSec}] EngStTime: [{trainEngine.engineStartupTime}] EngDmgToSlow: [{trainEngine.engineDamageToSlow}]" +
            //    $" EngDmgTimeFrame: [{trainEngine.engineDamageTimeframe}] EngSlwTime: [{trainEngine.engineSlowedTime}] EngSlwMaxVel: [{trainEngine.engineSlowedMaxVel}]" +
            //    $" DriverProtDens: [{trainEngine.driverProtection.density}]";
            //Debug.LogWarning(info);

            if (caConfig.maxHealth > -1) {
                trainEngine.SetMaxHealth(caConfig.maxHealth);
                trainEngine.SetHealth(caConfig.maxHealth);
            }
            if (caConfig.engineForce > -1)
                trainEngine.engineForce = caConfig.engineForce;
            if (caConfig.maxSpeed > -1)
                trainEngine.maxSpeed = caConfig.maxSpeed;
            if (caConfig.maxFuelPerSec > -1)
                trainEngine.maxFuelPerSec = caConfig.maxFuelPerSec;
            if (caConfig.idleFuelPerSec > -1)
                trainEngine.idleFuelPerSec = caConfig.idleFuelPerSec;
            if (caConfig.engineStartupTime > -1)
                trainEngine.engineStartupTime = caConfig.engineStartupTime;
            if (caConfig.engineDamageToSlow > -1)
                trainEngine.engineDamageToSlow = caConfig.engineDamageToSlow;
            if (caConfig.engineDamageTimeframe > -1)
                trainEngine.engineDamageTimeframe = caConfig.engineDamageTimeframe;
            if (caConfig.engineSlowedTime > -1)
                trainEngine.engineSlowedTime = caConfig.engineSlowedTime;
            if (caConfig.engineSlowedMaxVel > -1)
                trainEngine.engineSlowedMaxVel = caConfig.engineSlowedMaxVel;
            if (caConfig.protDensity > -1)
                trainEngine.driverProtection.density = caConfig.protDensity;

            if (caConfig.Slots > 1) {
                StorageContainer storage = (trainEngine.engineController.FuelSystem as EntityFuelSystem).GetFuelContainer();
                storage.panelName = "generic";
                storage.inventory.capacity = caConfig.Slots;

                if (caConfig.Store) {
                    storage.allowedItem = null;
                    storage.inventory.onlyAllowedItems = new ItemDefinition[0];
                }
            }
        }

        private void SetupCart(TrainCar train) {
            if (train == null) return;
            TrainEngine trainEngine = train?.GetComponent<TrainEngine>();
            train.FrontTrackSection.isStation = true;
            //trainEngine.CancelInvoke(trainEngine.DecayTick);

            if (Settings.TerrColl) {
                foreach (TriggerTrainCollisions Trigger in train.GetComponentsInChildren<TriggerTrainCollisions>()) {
                    Trigger.triggerCollider.gameObject.layer = passLayer;
                }
            }

            CustomWK caConfig; 
            Settings.WKPrefabs.Prefabs.TryGetValue(train.PrefabName, out caConfig);
            if (caConfig != null && trainEngine != null && caConfig.Enabled) SetupCustomisation(trainEngine, caConfig);
        }

        private void RespawnWorkCart(GameObject Track) {
            Vector3 atPos = Track.transform.position + Track.transform.forward * Settings.Distance; atPos.y += 0.1f;
            int index = Random.Range(1, Settings.WKPrefabs.Prefabs.Count+1) - 1;
            string caPrefab = Settings.WKPrefabs.Prefabs.ElementAt(index).Key;
            TrainEngine workCart = GameManager.server.CreateEntity(caPrefab, atPos, Track.transform.rotation) as TrainEngine;
            workCart?.Spawn(); TrackObjects[Track] = workCart;
        }
        #endregion

        #region Hooks

        object CanUseFuel(EntityFuelSystem fuelSystem, StorageContainer fuelContainer, float seconds, float fuelUsedPerSecond) {
            if (!Settings.CustomFuel) return null;

            GameObject Track = TrackObjects.FirstOrDefault(x => x.Value == fuelSystem.fuelStorageInstance.Get(true)?.GetParentEntity()).Key;
            if (Track.IsUnityNull() || fuelContainer.IsUnityNull()) return null;

            Item slot = fuelContainer?.inventory?.FindItemByItemID(LowGradeFuel);
            if (slot == null || slot.amount < 1) return 0;

            fuelSystem.pendingFuel += seconds * fuelUsedPerSecond;
            if (fuelSystem.pendingFuel >= 1f) {
                int num = Mathf.FloorToInt(fuelSystem.pendingFuel);
                slot.UseItem(num);
                fuelSystem.pendingFuel -= num;
                return num;
            }

            return 0;
        }

        object OnFuelItemCheck(EntityFuelSystem fuelSystem, StorageContainer fuelContainer) {
            if (!Settings.CustomFuel) return null;

            GameObject Track = TrackObjects.FirstOrDefault(x => x.Value == fuelSystem.fuelStorageInstance.Get(true)?.GetParentEntity()).Key;
            if (Track.IsUnityNull() || fuelContainer.IsUnityNull()) return null;

            List<Item> totItems = fuelContainer?.inventory?.FindItemsByItemID(LowGradeFuel);
            Item item;

            if (totItems == null) {
                item = ItemManager.CreateByItemID(LowGradeFuel);
                item.amount = 0;
            } else {
                item = ItemManager.CreateByItemID(LowGradeFuel); item.amount -= 1;
                totItems.ForEach(itm => { if (itm != item) item.amount += itm.amount; });
            }

            return item;
        }

        void OnEntitySpawned(TrainCar entity) {
            if (entity == null) return;
            NextTick(() => SetupCart(entity));
        }

        void OnEntityKill(TrainEngine entity, HitInfo info) {
            if (entity == null) return;

            GameObject Track = TrackObjects.FirstOrDefault(x => x.Value == entity).Key;
            if (Track != null) {
                KillTimer(Track);
                Timers.Add(Track, timer.Once(Settings.Time, () => RespawnWorkCart(Track)));
            }
        }
        #endregion

        #region Helpers
        void KillTimer(GameObject Track) {
            Timer timer; Timers.TryGetValue(Track, out timer);
            if (timer == null) return;
            timer.Destroy();
            Timers.Remove(Track);
        }

        public static int ToLayer(int bitmask) {
            int result = bitmask > 0 ? 0 : 31;
            while (bitmask > 1) {
                bitmask = bitmask >> 1;
                result++;
            }
            return result;
        }
        #endregion
    }
}
